//////////////////////////////////////////////////////////////////////////////
//  Copyright 2014 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.video - Implements video playback
 */
 
 // Namespace
var ntp = ntp || {};

ntp.video = ntp.video || {};

(function () {
    var videoPlayer = null; // The MediaElementPlayer instance.
    var $videoPlayerLightbox = $('#video_player_lightbox');
    
    var currentId = null;
    
    $.extend(ntp.video, {
        
        videoWidth: 800,
        videoHeight: 450,
                
        play: function () {
            if (videoPlayer) {
                videoPlayer.load();
                videoPlayer.play();
            }
        },
        stop: function () {
            if (videoPlayer) {
                videoPlayer.pause();
                
                // Restore the status of video controls including the close button
                // so that the lightbox will be in the same state if it is shown again.
                ntp.video.controls.show();
            }
        },

        /**
         * Stops video player in other new tab page instances. It works in this way:
         * When a new video starts playing, a new ID is generated to represent the
         * current video, and passed to AutoCAD. AutoCAD then invokes ntp.video.stopWithJSON(idJson)
         * in all new tab page instances with the current video ID. If the ID is different
         * from the currentId stored in a new tab page instance, the video in that instance
         * is stopped.
         */
        stopOthers: function () {
            this.currentId = new Date().getTime().toString();
            ntp.acad.stopVideoById(this.currentId);
        },
        
        stopWithJSON: function (idJSON) {
            var json = $.parseJSON(idJSON);
            if (json.id !== this.currentId) {
                this.stop();
            }
        },

        updateViewedState: function () {
            var $videoPlayer = $('#video_player');
            if (ntp.content.learnTab.urlIsHilightedContent($videoPlayer.data('url'))) {
                ntp.acad.updateLearnContentState($videoPlayer.data('url'), $videoPlayer.data('upload_time'));
            }
        },

        showPlayer: function ($item) {
            // Initialize video player if needed
            if (videoPlayer === null) {
                var $video = $('#video_player');        
                if ($video.size() == 0) {
                    // Create video element
                    $video = $('<video>').attr({
                        id: 'video_player',
                        preload: 'none',
                        type: 'video/webm'
                    });
                    $('#video_container').append($video);
                }
                
                $('#video_container').width(this.videoWidth);
            
                $video.attr({
                    src: $item.data('url'),
                    width: this.videoWidth,
                    height: this.videoHeight
                });
            
                // Enable/disable automatic hiding player controls
                $video.on('play', 
                    ntp.utils.makeCallback(this.controls.autohide, 'enable'));
                $video.on('pause error ended',
                    ntp.utils.makeCallback(this.controls.autohide, 'disable'));
                
                // Stop video player in other new tab page instances
                $video.on('play', ntp.utils.makeCallback(this, 'stopOthers'));

                //Once Played mark it as a viewed item
                $video.on('play', ntp.utils.makeCallback(this, 'updateViewedState'));

                // Start playing video to play the video in autoplay mode
                videoPlayer = new MediaElementPlayer('#video_player', {
                    pluginWidth: this.videoWidth,
                    pluginHeight: this.videoHeight,
                    alwaysShowControls: true,
                    hideVolumeOnTouchDevices: false
                });
                
                $videoPlayerLightbox.lightbox('bind', 'lightboxshown', 
                    ntp.utils.makeCallback(this, 'play'));
            
                $videoPlayerLightbox.lightbox('bind', 'lightboxhidden',
                    ntp.utils.makeCallback(this, 'stop'));
            
                // Disable activating video control buttons with tab key
                $('#video_container button').attr('tabindex', '-1');
                // Required for 'click-to-pause' to work inside AutoCAD
                $('.mejs-mediaelement').click(ntp.utils.makeCallback(this, 'stop'));
            } else {
                // Update existing video player
                videoPlayer.pause();
                videoPlayer.setSrc($item.data('url'));
            }
        
            //Add some data to query in updateViewedState callback
            $('#video_player').data({
                url: $item.data('url'),
                upload_time: $item.data('upload_time')
            });

            // Update video player title and prev/next links
            $('#video_container .title').text($item.data('title'));
            var $prevLink = $('#video_container .prev');
            $prevLink.off('click');
            var $prevItem = $item.prev();
            if ($prevItem.size() > 0) {
                $prevLink.children('span').text($prevItem.data('title'));
                $prevLink.show();
                $prevLink.on('click', 
                    ntp.utils.makeCallbackWithArgs(this, 'showPlayer', $prevItem));
            } else {
                $prevLink.hide();
            }
        
            var $nextLink = $('#video_container .next');
            $nextLink.off('click');
            var $nextItem = $item.next();
            if ($nextItem.size() > 0) {
                $nextLink.children('span').text($nextItem.data('title'));
                $nextLink.show();
                $nextLink.on('click',
                    ntp.utils.makeCallbackWithArgs(this, 'showPlayer', $nextItem));
            } else {
                $nextLink.hide();
            }
        
            // Play video
            if ($videoPlayerLightbox.lightbox('isShowing')) {
                videoPlayer.load();
                videoPlayer.play();
            } else {
                $videoPlayerLightbox.lightbox();
            }
        }
    });
    
})();

/**
 * Manipulation of video player controls
 */
ntp.video.controls = (function () {
    var isVisible = true;
    var updateVisibility = function (visible) {
        if (visible !== isVisible) {
            if (visible) {
                $('#video_container .mejs-controls')
                    .add('#video_container header')
                    .add('#lightbox_panel .close')
                    .stop(true, true).css('opacity', 1); // Use opacity to avoid layout issues.
            } else {
                $('#video_container .mejs-controls')
                        .add('#video_container header')
                        .add('#lightbox_panel .close')
                        .stop(true, true).delay(1000).fadeTo(400, 0);
            }
            isVisible = visible;
        }
    };
    
    return {
        show: function () {
            updateVisibility(true);
        },
        hoverArea: function () {
            return $('#video_container')
                .add($('#lightbox_panel .close'));
        },
        hovering: function () {
            var isHovering = false;
            this.hoverArea().each(function () {
                isHovering = $(this).is(':hover');
                return !isHovering; // Break when any element is :hover
            });
            return isHovering;
        },
        refresh: function () {
            updateVisibility(this.hovering());
        }
    };    
})();

/**
 * Automatically hide video player controls when user is not hovering over the player
 * area.
 */
ntp.video.controls.autohide = (function () {
    var isEnabled = false;    
    return {
        enable: function () {
            if (!isEnabled) {
                if ($('#video_container .me-cannotplay').size() > 0) {
                    // Disallow enabling auto hide if video cannot be played
                    return;
                }
                ntp.video.controls.refresh();
                ntp.video.controls.hoverArea()
                    .on('mouseenter mouseleave', 
                        ntp.utils.makeCallback(ntp.video.controls, 'refresh'));
                isEnabled = true;
            }
        },
        disable: function () {
            if (isEnabled) {
                ntp.video.controls.hoverArea()
                    .off('mouseenter mouseleave');            
                ntp.video.controls.show();
                isEnabled = false;
            }
        }
    };
})();

// SIG // Begin signature block
// SIG // MIIZ7QYJKoZIhvcNAQcCoIIZ3jCCGdoCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // gSmWRtJg5qTgMwAnPK46dqGLWFLVV5CrZIUhN9/LOByg
// SIG // ggpjMIIFAjCCA+qgAwIBAgIQZLyxx87k2lhp1Y8eIvvp
// SIG // 9DANBgkqhkiG9w0BAQsFADB/MQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAd
// SIG // BgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxMDAu
// SIG // BgNVBAMTJ1N5bWFudGVjIENsYXNzIDMgU0hBMjU2IENv
// SIG // ZGUgU2lnbmluZyBDQTAeFw0xNTA5MDMwMDAwMDBaFw0x
// SIG // NjA5MDIyMzU5NTlaMIGIMQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwKU2FuIFJh
// SIG // ZmFlbDEWMBQGA1UECgwNQXV0b2Rlc2ssIEluYzEfMB0G
// SIG // A1UECwwWRGVzaWduIFNvbHV0aW9ucyBHcm91cDEWMBQG
// SIG // A1UEAwwNQXV0b2Rlc2ssIEluYzCCASIwDQYJKoZIhvcN
// SIG // AQEBBQADggEPADCCAQoCggEBAOqZ9OjPzARqd9dP4fq2
// SIG // qXdpTJolFYJ8Wxg8WcZVciX3l+dljew4R9C8WR/0tW17
// SIG // lLANaSORF4IYmWTgdpFg5PGR3s+R4A461hpRIG9QthGU
// SIG // q8Bt7af5VbCjjKtYBNq/x3ukqVsw/1/qljGVplDOyr97
// SIG // 6ktnCm0/wL0N+ubk5WAIgv+I9E0i6+GX1hRm+eRliYKe
// SIG // KF0/gBLyDh3ut+N3HNPrnfjfec7q2Czd82CeRBCToR1E
// SIG // 7Cp6yBBdGUfb3JE0PnE1+6n0qJnsvs2bsCIzqVo3A0iM
// SIG // HBsbioxa5DYyXuEfNCktosUlFXfegd514ZJns6YDr4Pl
// SIG // ahC0+lK3hgJy5vkCAwEAAaOCAW4wggFqMAkGA1UdEwQC
// SIG // MAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMDMGYGA1UdIARfMF0wWwYLYIZIAYb4RQEHFwMw
// SIG // TDAjBggrBgEFBQcCARYXaHR0cHM6Ly9kLnN5bWNiLmNv
// SIG // bS9jcHMwJQYIKwYBBQUHAgIwGRoXaHR0cHM6Ly9kLnN5
// SIG // bWNiLmNvbS9ycGEwHwYDVR0jBBgwFoAUljtT8Hkzl699
// SIG // g+8uK8zKt4YecmYwKwYDVR0fBCQwIjAgoB6gHIYaaHR0
// SIG // cDovL3N2LnN5bWNiLmNvbS9zdi5jcmwwVwYIKwYBBQUH
// SIG // AQEESzBJMB8GCCsGAQUFBzABhhNodHRwOi8vc3Yuc3lt
// SIG // Y2QuY29tMCYGCCsGAQUFBzAChhpodHRwOi8vc3Yuc3lt
// SIG // Y2IuY29tL3N2LmNydDARBglghkgBhvhCAQEEBAMCBBAw
// SIG // FgYKKwYBBAGCNwIBGwQIMAYBAQABAf8wDQYJKoZIhvcN
// SIG // AQELBQADggEBAB6BYdY8nzLWS3nsk/xM5CWdizR4wyDn
// SIG // 8wfGVbcuEq/88byNnY0ODxisa7AELvLe6b48DorFFP2n
// SIG // Hw0qOeIxfg2w+1DzWzYQhMfLi4V/Pp1OmCYvYEsTb/oU
// SIG // +YbiSgSFjrN5Pi2LN5Oxw5Ey7WAK2vhWr4EL/7QDHoso
// SIG // 4a3lS3WklPz2K/Nlhk4i8S230BlGCHsZGDY7PtfowWdz
// SIG // cq50xCEIScERSRtLd+YL6SWXk+6gX9HVcFxtvLQ4N9Df
// SIG // F/BKk13pYgyvZPkedIKA4yYCy4/VjouNoKhgM/aaGLAy
// SIG // Dv3ujRdXLCcosR4MbD8X1v9z6WkgG5YZZicWPPsJN7ty
// SIG // B1wwggVZMIIEQaADAgECAhA9eNf5dklgsmF99PAeyoYq
// SIG // MA0GCSqGSIb3DQEBCwUAMIHKMQswCQYDVQQGEwJVUzEX
// SIG // MBUGA1UEChMOVmVyaVNpZ24sIEluYy4xHzAdBgNVBAsT
// SIG // FlZlcmlTaWduIFRydXN0IE5ldHdvcmsxOjA4BgNVBAsT
// SIG // MShjKSAyMDA2IFZlcmlTaWduLCBJbmMuIC0gRm9yIGF1
// SIG // dGhvcml6ZWQgdXNlIG9ubHkxRTBDBgNVBAMTPFZlcmlT
// SIG // aWduIENsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlm
// SIG // aWNhdGlvbiBBdXRob3JpdHkgLSBHNTAeFw0xMzEyMTAw
// SIG // MDAwMDBaFw0yMzEyMDkyMzU5NTlaMH8xCzAJBgNVBAYT
// SIG // AlVTMR0wGwYDVQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlv
// SIG // bjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29y
// SIG // azEwMC4GA1UEAxMnU3ltYW50ZWMgQ2xhc3MgMyBTSEEy
// SIG // NTYgQ29kZSBTaWduaW5nIENBMIIBIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAQ8AMIIBCgKCAQEAl4MeABavLLHSCMTXaJNR
// SIG // YB5x9uJHtNtYTSNiarS/WhtR96MNGHdou9g2qy8hUNqe
// SIG // 8+dfJ04LwpfICXCTqdpcDU6kDZGgtOwUzpFyVC7Oo9tE
// SIG // 6VIbP0E8ykrkqsDoOatTzCHQzM9/m+bCzFhqghXuPTbP
// SIG // HMWXBySO8Xu+MS09bty1mUKfS2GVXxxw7hd924vlYYl4
// SIG // x2gbrxF4GpiuxFVHU9mzMtahDkZAxZeSitFTp5lbhTVX
// SIG // 0+qTYmEgCscwdyQRTWKDtrp7aIIx7mXK3/nVjbI13Iwr
// SIG // b2pyXGCEnPIMlF7AVlIASMzT+KV93i/XE+Q4qITVRrgT
// SIG // hsIbnepaON2b2wIDAQABo4IBgzCCAX8wLwYIKwYBBQUH
// SIG // AQEEIzAhMB8GCCsGAQUFBzABhhNodHRwOi8vczIuc3lt
// SIG // Y2IuY29tMBIGA1UdEwEB/wQIMAYBAf8CAQAwbAYDVR0g
// SIG // BGUwYzBhBgtghkgBhvhFAQcXAzBSMCYGCCsGAQUFBwIB
// SIG // FhpodHRwOi8vd3d3LnN5bWF1dGguY29tL2NwczAoBggr
// SIG // BgEFBQcCAjAcGhpodHRwOi8vd3d3LnN5bWF1dGguY29t
// SIG // L3JwYTAwBgNVHR8EKTAnMCWgI6Ahhh9odHRwOi8vczEu
// SIG // c3ltY2IuY29tL3BjYTMtZzUuY3JsMB0GA1UdJQQWMBQG
// SIG // CCsGAQUFBwMCBggrBgEFBQcDAzAOBgNVHQ8BAf8EBAMC
// SIG // AQYwKQYDVR0RBCIwIKQeMBwxGjAYBgNVBAMTEVN5bWFu
// SIG // dGVjUEtJLTEtNTY3MB0GA1UdDgQWBBSWO1PweTOXr32D
// SIG // 7y4rzMq3hh5yZjAfBgNVHSMEGDAWgBR/02Wnwt3su/Aw
// SIG // CfNDOfoCrzMxMzANBgkqhkiG9w0BAQsFAAOCAQEAE4Ua
// SIG // HmmpN/egvaSvfh1hU/6djF4MpnUeeBcj3f3sGgNVOftx
// SIG // lcdlWqeOMNJEWmHbcG/aIQXCLnO6SfHRk/5dyc1eA+CJ
// SIG // nj90Htf3OIup1s+7NS8zWKiSVtHITTuC5nmEFvwosLFH
// SIG // 8x2iPu6H2aZ/pFalP62ELinefLyoqqM9BAHqupOiDlAi
// SIG // KRdMh+Q6EV/WpCWJmwVrL7TJAUwnewusGQUioGAVP9rJ
// SIG // +01Mj/tyZ3f9J5THujUOiEn+jf0or0oSvQ2zlwXeRAwV
// SIG // +jYrA9zBUAHxoRFdFOXivSdLVL4rhF4PpsN0BQrvl8OJ
// SIG // IrEfd/O9zUPU8UypP7WLhK9k8tAUITGCDuIwgg7eAgEB
// SIG // MIGTMH8xCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1h
// SIG // bnRlYyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50
// SIG // ZWMgVHJ1c3QgTmV0d29yazEwMC4GA1UEAxMnU3ltYW50
// SIG // ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
// SIG // AhBkvLHHzuTaWGnVjx4i++n0MA0GCWCGSAFlAwQCAQUA
// SIG // oHwwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcNAQkD
// SIG // MQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwG
// SIG // CisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIL5cAcg+
// SIG // 9yIT+hSzJhVI9aZIfQZhuXIv7s2ki1Tlpp2hMA0GCSqG
// SIG // SIb3DQEBAQUABIIBADMlZOoL02UhWcQAy/Wge1zHb5z5
// SIG // dKyOokZAwxI3IJ9FbYB9pqP2lZvZ6CB0GfJhK2PoSZdp
// SIG // gW8okFgbM4aMBfaRr1dC0NNEayXEtlQiCPWd0AXeefMR
// SIG // PKTKGjEG9+00vORpG2yxfYbq9EZ52h81m4JLMiEifkLY
// SIG // SNXXWlhH0bBbMsq/zEnGQTsvC9+Ofu+VyMgK+uUqJwzM
// SIG // awyJRg+8l10msrMedn1cUruXmCauGD/w9eaoLD/JVvg/
// SIG // 33Yot5gSXrTgUCZWDQe75ak+YP+k9Iah8iswmY1LUA+H
// SIG // ZwcPAWhMh6khSvs07R+tJ4pmlVKKYhg5vyYyQ73EaOxu
// SIG // eMhAPCyhggyhMIIMnQYKKwYBBAGCNwMDATGCDI0wggyJ
// SIG // BgkqhkiG9w0BBwKgggx6MIIMdgIBAzEPMA0GCWCGSAFl
// SIG // AwQCAQUAMIHdBgsqhkiG9w0BCRABBKCBzQSByjCBxwIB
// SIG // AQYJKwYBBAGgMgIDMDEwDQYJYIZIAWUDBAIBBQAEILj4
// SIG // VWdqIt9s1DzQuD4Pl/z38BEewDfkLTklnzAA2rN/AhQO
// SIG // vjPoyh74frotPaJgtUN6V3OArBgPMjAxNjAyMDcwNDM3
// SIG // NDVaoF2kWzBZMQswCQYDVQQGEwJTRzEfMB0GA1UEChMW
// SIG // R01PIEdsb2JhbFNpZ24gUHRlIEx0ZDEpMCcGA1UEAxMg
// SIG // R2xvYmFsU2lnbiBUU0EgZm9yIEFkdmFuY2VkIC0gRzKg
// SIG // ggjGMIIEqTCCA5GgAwIBAgISESEWwAmY3MaPon0lw4Y2
// SIG // qIO7MA0GCSqGSIb3DQEBCwUAMFsxCzAJBgNVBAYTAkJF
// SIG // MRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYD
// SIG // VQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAt
// SIG // IFNIQTI1NiAtIEcyMB4XDTE1MDIwMzAwMDAwMFoXDTI2
// SIG // MDMwMzAwMDAwMFowWTELMAkGA1UEBhMCU0cxHzAdBgNV
// SIG // BAoTFkdNTyBHbG9iYWxTaWduIFB0ZSBMdGQxKTAnBgNV
// SIG // BAMTIEdsb2JhbFNpZ24gVFNBIGZvciBBZHZhbmNlZCAt
// SIG // IEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC
// SIG // AQEAt8eSiijY85HLUA1d6ylEwYJZpcpkzr2DtVHllMxu
// SIG // gJoXbNz/YuY2Q14Bwgxkjcp3nvZdd0W5Xsgk60E3EWY3
// SIG // rMjNdJvMs+9M4n8bznO+yZadNR5891T3bOygEtaumZ1j
// SIG // xHhfWakuIvXDJ9iyW0VjJ5D/jLE9qbTzDsDxIGUGiZ0D
// SIG // HKOwUkGFNOu8y85U9hL+swGJSqb6VMUf+9McwOO/l+M/
// SIG // 0Bv5QMcb1esmfEMwx3DyeBGqhU6lDg0ASjoRd6PKIwGA
// SIG // nsiTVl1GzIrY0qoKhIWyiVvVMs5BexhOVMqr0VJeEtMk
// SIG // nlbzDI/dvZKOedVBicOjoOWR/MdnnYczznEycwIDAQAB
// SIG // o4IBZzCCAWMwDgYDVR0PAQH/BAQDAgeAMEwGA1UdIARF
// SIG // MEMwQQYJKwYBBAGgMgEeMDQwMgYIKwYBBQUHAgEWJmh0
// SIG // dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3JlcG9zaXRv
// SIG // cnkvMAkGA1UdEwQCMAAwFgYDVR0lAQH/BAwwCgYIKwYB
// SIG // BQUHAwgwRgYDVR0fBD8wPTA7oDmgN4Y1aHR0cDovL2Ny
// SIG // bC5nbG9iYWxzaWduLmNvbS9ncy9nc3RpbWVzdGFtcGlu
// SIG // Z3NoYTJnMi5jcmwwWAYIKwYBBQUHAQEETDBKMEgGCCsG
// SIG // AQUFBzAChjxodHRwOi8vc2VjdXJlLmdsb2JhbHNpZ24u
// SIG // Y29tL2NhY2VydC9nc3RpbWVzdGFtcGluZ3NoYTJnMi5j
// SIG // cnQwHQYDVR0OBBYEFC1ubtGN5QOA7udj6afZ2gs8VyI9
// SIG // MB8GA1UdIwQYMBaAFJIhp0qVXWSwm7Qe5gA3R+adQStM
// SIG // MA0GCSqGSIb3DQEBCwUAA4IBAQB/CgOhVXb+qvoU0RF6
// SIG // 4uIPOjL/e0bGBHGMXtDMeUfddQdtuki+P2DiyxEgwrar
// SIG // Q8e3otm4Wv7NtDqz8VJTDLSO/2LdiJuyGn8bUbdxh9pD
// SIG // SzdSzf4+BilwpCds59w8f3bjkMga6ZkSUO7qTCMayF+c
// SIG // 1SmEKuaBYgZ2gnajE1UOSKZR0fNMDTwIICiGaoO3nQyg
// SIG // XmA0BjGofXGPUfIE36Imk7nVKHejoLdCo4swJ1oJ9tyD
// SIG // mCHVfUGaBxkUWih2fa2goyQBTuEWyL9/R/M6BF5lPqJM
// SIG // //1vBs+suBsAZ/rva37X4gq/e8WnEqzXNPVXiD6Elq1r
// SIG // VC49Jw/iORrBxxhyMIIEFTCCAv2gAwIBAgILBAAAAAAB
// SIG // MYnGUAQwDQYJKoZIhvcNAQELBQAwTDEgMB4GA1UECxMX
// SIG // R2xvYmFsU2lnbiBSb290IENBIC0gUjMxEzARBgNVBAoT
// SIG // Ckdsb2JhbFNpZ24xEzARBgNVBAMTCkdsb2JhbFNpZ24w
// SIG // HhcNMTEwODAyMTAwMDAwWhcNMjkwMzI5MTAwMDAwWjBb
// SIG // MQswCQYDVQQGEwJCRTEZMBcGA1UEChMQR2xvYmFsU2ln
// SIG // biBudi1zYTExMC8GA1UEAxMoR2xvYmFsU2lnbiBUaW1l
// SIG // c3RhbXBpbmcgQ0EgLSBTSEEyNTYgLSBHMjCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAKqbjsOrEVEl
// SIG // AbaWlOJP2MEI9kYj2UXFlZdbqxq/0mxXyTMGH6APxjx+
// SIG // U0h6v52Hnq/uw4xH4ULs4+OhSmwMF8SmwbnNW/EeRImO
// SIG // /gveIVgT7k3IxWcLHLKz8TR2kaLLB203xaBHJgIVpJCR
// SIG // qXme1+tXnSt8ItgU1/EHHngiNmt3ea+v+X+OTuG1CDH9
// SIG // 6u1LcWKMI/EDOY9EebZ2A1eerS8IRtzSjLz0jnTOyGhp
// SIG // UXYRiw9dJFsZVD0mzECNgicbWSB9WfaTgI74Kjj9a6BA
// SIG // ZR9XdsxbjgRPLKjbhFATT8bci7n43WlMiOucezAm/HpY
// SIG // u1m8FHKSgVe3dsnYgAqAbgkCAwEAAaOB6DCB5TAOBgNV
// SIG // HQ8BAf8EBAMCAQYwEgYDVR0TAQH/BAgwBgEB/wIBADAd
// SIG // BgNVHQ4EFgQUkiGnSpVdZLCbtB7mADdH5p1BK0wwRwYD
// SIG // VR0gBEAwPjA8BgRVHSAAMDQwMgYIKwYBBQUHAgEWJmh0
// SIG // dHBzOi8vd3d3Lmdsb2JhbHNpZ24uY29tL3JlcG9zaXRv
// SIG // cnkvMDYGA1UdHwQvMC0wK6ApoCeGJWh0dHA6Ly9jcmwu
// SIG // Z2xvYmFsc2lnbi5uZXQvcm9vdC1yMy5jcmwwHwYDVR0j
// SIG // BBgwFoAUj/BLf6guRSSuTVD6Y5qL3uLdG7wwDQYJKoZI
// SIG // hvcNAQELBQADggEBAARWgkp80M7JvzZm0b41npNsl+gG
// SIG // zjEYWflsQV+ALsBCJbgYx/zUsTfEaKDPKGoDdEtjl4V3
// SIG // YTvXL+P1vTOikn0RH56KbO8ssPRijTZz0RY28bxe7LSA
// SIG // mHj80nZ56OEhlOAfxKLhqmfbs5xz5UAizznO2+Z3lae7
// SIG // ssv2GYadn8jUmAWycW9Oda7xPWRqO15ORqYqXQiS8aPz
// SIG // HXS/Yg0jjFwqOJXSwNXNz4jaHyi1uoFpZCq1pqLVc6/c
// SIG // RtsErpHXbsWYutRHxFZ0gEd4WIy+7yv97Gy/0ZT3v1Dg
// SIG // e+CQ/SAYeBgiXQgujBygl/MdmX2jnZHTBkROBG56HCDj
// SIG // NvC2ULkxggK0MIICsAIBATBxMFsxCzAJBgNVBAYTAkJF
// SIG // MRkwFwYDVQQKExBHbG9iYWxTaWduIG52LXNhMTEwLwYD
// SIG // VQQDEyhHbG9iYWxTaWduIFRpbWVzdGFtcGluZyBDQSAt
// SIG // IFNIQTI1NiAtIEcyAhIRIRbACZjcxo+ifSXDhjaog7sw
// SIG // DQYJYIZIAWUDBAIBBQCgggEUMBoGCSqGSIb3DQEJAzEN
// SIG // BgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMTYw
// SIG // MjA3MDQzNzQ1WjAvBgkqhkiG9w0BCQQxIgQg6rS9g/zz
// SIG // 665doDlAILjTYG8G1wAxdXX4RxmOexAAjykwgaYGCyqG
// SIG // SIb3DQEJEAIMMYGWMIGTMIGQMIGNBBRXqt6jTjqEJxGX
// SIG // sll4jXMMauIuyTB1MF+kXTBbMQswCQYDVQQGEwJCRTEZ
// SIG // MBcGA1UEChMQR2xvYmFsU2lnbiBudi1zYTExMC8GA1UE
// SIG // AxMoR2xvYmFsU2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBT
// SIG // SEEyNTYgLSBHMgISESEWwAmY3MaPon0lw4Y2qIO7MA0G
// SIG // CSqGSIb3DQEBAQUABIIBAJbOgULqX7HmFJCUv39U9ow5
// SIG // r5Slst5jNLWBnfGgzhKBbBqZyLRPIR9eBG1UTSxPt63H
// SIG // lbWGO5pOtPG/elYJIoQZ0OL5NxPokL5yfQ9X9J1QFSen
// SIG // lyhDd29bFoeJTbwhc+s/HALOVMIpr82TeY0avw2Lhk+p
// SIG // tKEbH2N5PViAboa0/sEM29dCtPXAsekQvQ3hsIWSQ1ph
// SIG // e5A7EcwydxuWGsPvog+8pOYMqKlqP9tdy569+6Up19yh
// SIG // 7zeK8D6TnpUgI9O/ewg4TgkwHzrfD6LQzrUjHYz1sKYj
// SIG // pHEBSqk8mjdw1L6PAUNScQ2YLFPLVw6JiimkMvupGSTZ
// SIG // K3OpfP2Hg8c=
// SIG // End signature block
